using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;

using VirtualLab.Programming;
using VirtualLab.Design;
using VirtualLabAPI.Core.BasicFunctions;
using VirtualLabAPI.Core.Common;
using VirtualLabAPI.Core.DataVisualization;
using VirtualLabAPI.Core.FieldRepresentations;
using VirtualLabAPI.Core.Functions;
using VirtualLabAPI.Core.GeometryDescription;
using VirtualLabAPI.Core.LightPath;
using VirtualLabAPI.Core.Materials;
using VirtualLabAPI.Core.Modules;
using VirtualLabAPI.Core.Numerics;
using VirtualLabAPI.Core.Numerics.Region2D;
using VirtualLabAPI.Core.OpticalSystems;
using VirtualLabAPI.Core.Propagation;
using VirtualLabAPI.Core.SupportFunctions;

namespace OwnCode {
    public class VLModule : IVLModule {
        public void Run() {
            
            // get input function
            DataArray1D inputFunction = (DataArray1D) VL_GUI.SelectOpenDataArray1D();
            // consistency check
            if (inputFunction.Data.Length != 1)
            {
                Globals.DataDisplay.LogError("Input DataArray1D contains more than one layer of data.");
            }
            else if(inputFunction.IsEquidistant == false)
            {
                Globals.DataDisplay.LogError("Input data is not equidistant.");
            }
            else
            {
                // extract data 
                ComplexField dataContainer = inputFunction.Data[0];
                // consistency check
                if(dataContainer.SamplingPoints.Y != 1 && dataContainer.SamplingPoints.X != 1)
                {
                    Globals.DataDisplay.LogError("Input data is not one-dimensional.");
                }
                else
                {
                    // get length of data
                    int LengthOfData;
                    if(dataContainer.SamplingPoints.X == 1)
                    {
                        LengthOfData = dataContainer.SamplingPoints.Y;
                    }
                    else
                    {
                        LengthOfData = dataContainer.SamplingPoints.X;
                    }
                    // initialize output data container
                    // save coordinate values into a "ComplexField" container 
                    // note that "ComplexField" is used only as a container, but not necessarily represent any "Field"
                    ComplexField t = new ComplexField(new Vector(1, LengthOfData),
                        false,
                        PrecisionMode.Double);
                    // also do the same for the data values
                    ComplexField f = new ComplexField(new Vector(1, LengthOfData),
                        false,
                        PrecisionMode.Double);
                    // fill t and f values
                    for(int i = 0; i < LengthOfData; i++)
                    {
                        // find coordinate t
                        double tValue = inputFunction.CoordinateOfFirstDataPoint + i * inputFunction.SamplingDistance;
                        // find f(t)
                        double fValue;
                        if(dataContainer.SamplingPoints.X == 1)
                        {
                            fValue = dataContainer[0, i].Re;
                        }
                        else
                        {
                            fValue = dataContainer[i, 0].Re;
                        }
                        // fill the values
                        t[0, i] = tValue;
                        f[0, i] = fValue;
                    }
                    // generate data array 
                    DataArray1D outputDataArray = new DataArray1D(new ComplexField1DArray(new ComplexField[1] { t }),
                        new PhysicalProperty[1] { inputFunction.PhysicalPropertyOfCoordinates }, // physical property of the two functions
                        new string[1] { inputFunction.CommentOfCoordinates }, // name of the two functions
                        new CFieldDerivative1DReal(f), // note that the "CFieldDerivative1DReal" is used only as a data container
                        inputFunction.PhysicalPropertiesOfDataEntries[0], // physical property of the variable - t
                        inputFunction.CommentsOfDataEntries[0], // name of the variable
                        f[0, LengthOfData-1].Re + f[0, LengthOfData-1].Re - f[0, LengthOfData-2].Re);
                    // display
                    Globals.DataDisplay.ShowDocument(outputDataArray, "Inversed Function");
                }
            }
            
            
        }
    }
}